// --------------------------------- LPShid.h -------------------------------------------
//
//	Include file for Linux LabBrick phase shifter API V1.11
//
// (c) 2011-2020 by Vaunix Technology Corporation, all rights reserved
//
//	HME Version 1.0 based on RD Version 1.0 for Windows
//	RD	updated for LDA-302 family
//	RD	updated to support High Res LDA devices
//	RD	updated for multiple channel devices
//	RD  added new device type value list, new API functions, and 8 channel support
//  RD	added support for new devices and new multi-channel API functions
//	RD	added support for new devices and expandable LDA devices
//  JA  added support for new device of LDA-608V-4
//  NB  adapted for LPS devices
//-----------------------------------------------------------------------------

//#include "libusb.h"
#include <libusb-1.0/libusb.h>

// Hardware type values by device name (used in lps[DeviceId].DevType)
#define LPS_802		1
#define LPS_123		2
#define LPS_402		3
#define LPS_202		4
#define LPS_802_4	5
#define LPS_802_8	6
#define LPS_402_4 7
#define LPS_402_8 8
#define LPS_202_8 9


#define VNX_MIN_DWELLTIME 1
#define STATUS_PROFILE_ACTIVE 0x80		// MASK: A profile is playing

// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)
#define SWP_DIRECTION		0x04		// MASK: bit = 0 for sweep or ramp up, 1 for sweep or ramp down
#define SWP_CONTINUOUS		0x02		// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE			0x01		// MASK: bit = 1 for single sweep
#define SWP_BIDIR			0x10		// MASK: bit = 0 for ramp style sweep, 1 for triangle style sweep

// ----------- Profile Control -----------
#define PROFILE_ONCE	1		// play the profile once
#define PROFILE_REPEAT	2		// play the profile repeatedly
#define PROFILE_OFF	0			// stop the profile

// HID report equates
#define HR_BLOCKSIZE 6			// size of the block of bytes buffer in our HID report


#define HID_REPORT_LENGTH 8 		// use an 8 byte report..

typedef struct
{
  char reportid;
  char status;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT1;

typedef struct
{
  char reportid;
  char command;
  char count;
  char byteblock[HR_BLOCKSIZE];
} HID_REPORT_OUT;

// Misc commands to send to the device
// For the input reports the first byte is the status, for the output it is the command. The high bit sets the
// direction.
//
//	count is the number of valid bytes in the byteblock array
// 	byteblock is an array of bytes which make up the value of the command's argument or arguments.
//
// For result reports the command portion of the status byte is equal to the command sent if the command was successful.
// status byte layout:

// Bit0 - Bit5 = command result, equal to command if everything went well
// Bit6 = --reserved--
// Bit7 = --reserved--

// All sweep related data items are DWORD (unsigned) quantities, stored in normal Microsoft byte order.
// Dwell time is a DWORD (unsigned)


// Misc commands to send to the device
#define VNX_SET				0x80
#define VNX_GET				0x00	// the set and get bits are or'd into the msb of the command byte


// ---------------------- Phase shifter commands ------------------------
#define VNX_PHASEANGLE		0x50	// phase shift angle in degrees (DWORD)
#define VNX_FREQUENCY		0x04	// working frequency in 100Khz units (set in 1MHz increments)
#define VNX_MINFREQUENCY	0x20	// minimum working frequency in 100Khz units (set in 1MHz increments)
#define VNX_MAXFREQUENCY	0x21	// maximum working frequency in 100Khz units (set in 1MHz increments)

// ----------------- Phase shifter ramp commands ------------------------
#define VNX_SWEEP			0x09	// command to start/stop sweep, data = 01 for single sweep, 00 to stop
									// sweeping, and 02 for continuous sweeping.

#define VNX_ASTART			0x30	// initial value for phase ramp
#define VNX_ASTOP			0x31	// final value for phase ramp

#define VNX_ASTEP			0x32	// step size for first part of phase ramp
#define VNX_ASTEP2			0x38	// step size for second part of phase ramp

#define VNX_ADWELL			0x33	// dwell time on each step for first part of phase ramp
#define VNX_ADWELL2			0x37	// dwell time on each step for second part of phase ramp

#define VNX_AIDLE			0x36	// idle time between phase ramps in milliseconds
#define VNX_AHOLD			0x39	// hold time between parts of the ramps in milliseconds

#define VNX_SETPROFILE		0x3A	// set/get profile values, first byte is unused
									// second data byte is the index (0 based)
									// the third is the angle value for that profile entry
									
#define VNX_PROFILECOUNT	0x3B	// number of elements in the profile, 1 to PROFILE_MAX = 100

#define VNX_PROFILEDWELL	0x3C	// dwell time for each profile element

#define VNX_PROFILEIDLE		0x3D	// idle time at the end of each repeating profile 

#define VNX_SAVEPAR			0x0C	// command to save user parameters to flash, data bytes must be
									// set to 0x42, 0x55, 0x31 as a key to enable the flash update
									// all of the above settings are saved (angle, sweep parameters, etc.)

#define VNX_MINPHASE		0x34	// used to get the minimum phase shift angle which is 0 for every case now

#define VNX_MAXPHASE		0x35	// get the maximum phase shift

#define VNX_GETSERNUM		0x1F	// get the serial number, value is a DWORD

#define VNX_MODELNAME		0x22	// get (no set allowed) the device's model name string -- last 6 chars only

#define VNX_DEFAULTS		0x0F	// restore all settings to factory default

// ------------------------ Multi-channel phase shifter Commands ----------------------------------
#define VNX_CHANNEL			0x54	// set the channel, 

//------------------------- Status Report ID Byte -------------------------------------
#define VNX_STATUS			0x52	// Not really a command, but the status byte value for periodic status reports.		

#define VNX_HRSTATUS		0x53	// status report used by HiRes  -- !!Check this, it may be wrong. Not used at the moment!!
#define VNX_HR8STATUS		0x55	// status report used by HiRes

// ----------- Global Equates ------------
#define MAXDEVICES 64
#define MAX_MODELNAME 32
#define PROFILE_MAX 50		// Older non-hires phase shifters can store 50 profile elements in their eeprom
#define PROFILE_MAX_RAM 1000	// New FW V2.x based phase shifters have a 1000 element RAM profile buffer
#define PROFILE_MAX_HR 25	// HiRes phase shifters only save 25 elements in their eeprom
#define CHANNEL_MAX 64		// The largest device today has 64 channels with a full set of expansion modules

// ----------- Data Types ----------------
#define DEVID unsigned int
#define PROFILE_EMPTY 0xFFFFFFFF // we use this as our marker for an empty entry in the profile

typedef struct
{
  //  Global device variables
  int DevType;
  int MinFrequency;
  int MaxFrequency;
  int MinPhase;
  int MaxPhase;			// maximum phase shift in .05 degree units
  int MinPhaseStep;				// replaces DevResolution, smallest phase step in .05 degree units
  int UnitScale;				// the number of .05 degree units in the device's HW representation (1 unit for .05 degrees, 20 units for 1 degree)
  int NumChannels;				// the number of channels the device has including expansion channels
  bool Expandable;				// true for devices that can have expansion channels
  int ProfileMaxLength;			// 25, 50, or 1000 depending on the device
  int ProfileMaxSaved;			// the maximum length of the profile that can be saved in eeprom, 25 for HiRes devices, 50 for other devices
  volatile int Channel;			// the current channel number
  volatile int ChMask;			// our channel in bitmask form (0x01 to 0x80 for single channels)
  volatile int GlobalChannel;	// the current global channel using a zero based index
  volatile int BaseChannels;	// the total number of channels in the base LPS device
  int SerialNumber;
  char ModelName[MAX_MODELNAME];
  unsigned int FrameNumber;
  //  Per channel variables
  volatile int DevStatus[CHANNEL_MAX];
  volatile int WorkingFrequency[CHANNEL_MAX];
  volatile int PhaseAngle[CHANNEL_MAX];				// in .05 degree units
  volatile int RampStart[CHANNEL_MAX];
  volatile int RampStop[CHANNEL_MAX];
  volatile int PhaseStep[CHANNEL_MAX];			// ramp step size for the first phase of the ramp
  volatile int PhaseStep2[CHANNEL_MAX];			// ramp step size for second phase of the ramp
  volatile int DwellTime[CHANNEL_MAX];
  volatile int DwellTime2[CHANNEL_MAX];
  volatile int IdleTime[CHANNEL_MAX];
  volatile int HoldTime[CHANNEL_MAX];
  volatile int ProfileCount[CHANNEL_MAX];
  volatile int ProfileIndex[CHANNEL_MAX];
  volatile int ProfileDwellTime[CHANNEL_MAX];
  volatile int ProfileIdleTime[CHANNEL_MAX];
  volatile int Modebits[CHANNEL_MAX];
  volatile int CachedProfileValue[CHANNEL_MAX];			// we only cache the last used profile entry now to avoid allocating a large but rarely used memory buffer
														// we don't use a malloc/free strategy to avoid memory leak issues if an application fails to close a device
														// The upper 16 bits holds the index, the lower 16 bits holds the value. Packing them as a pair into one int avoids possible race conditions
														// due to a non-atomic read of the structure if they were separate variables.

  // Internal variables used to identify and manage the hardware
  unsigned int idVendor;
  unsigned int idProduct;
  unsigned int idType;
  int BusAddress;
  int Endpoint;
  char Serialstr[16];
  volatile char thread_command;
  char sndbuff[8];
  char rcvbuff[24];
  volatile char decodewatch;
  int MyDevID;
  libusb_device_handle *DevHandle;

} LPSPARAMS;

// ----------- Mode Bit Masks ------------

#define MODE_SWEEP 		0x0000001F 	// bottom 5 bits are used to keep the sweep control bits

// ----------- Profile Control -----------
#define PROFILE_ONCE    1                // play the profile once
#define PROFILE_REPEAT  2                // play the profile repeatedly
#define PROFILE_OFF     0                // stop the profile

// ----------- Command Equates -----------

#define BYTE unsigned char

// Status returns for commands
#define LVSTATUS unsigned int

#define STATUS_OK 0
#define INVALID_DEVID       	0x80000000
#define BAD_PARAMETER  			  0x80010000 	// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO     			  0x80020000
#define DEVICE_NOT_READY  		0x80030000 	// device isn't open, no handle, etc.
#define FEATURE_NOT_SUPPORTED	0x80040000	// the selected Lab Brick does not support this function
                                          // Profiles and Bi-directional ramps are only supported in
                                          // newer LPS models
#define DATA_UNAVAILABLE      0x80050000  // the requested data item is unavailable
                                          // usually due to a ramp or profile where reporting times exceed timeout limits


// Some of these are global in that they describe the hardware and some are
// channel specific.

// Status returns for DevStatus
#define INVALID_DEVID 		0x80000000 	// MSB is set if the device ID is invalid [global]
#define DEV_CONNECTED 		0x00000001 	// LSB is set if a device is connected [global]
#define DEV_OPENED 			0x00000002 	// set if the device is opened [global]
#define SWP_ACTIVE 			0x00000004 	// set if the device is sweeping
#define SWP_UP 				0x00000008 	// set if the device is sweeping up in frequency
#define SWP_REPEAT 			0x00000010 	// set if the device is in continuous sweep mode
#define SWP_BIDIRECTIONAL	0x00000020	// set if the device is in bi-directional ramp mode
#define PROFILE_ACTIVE		0x00000040	// set if a profile is playing

// Internal values in DevStatus
#define DEV_LOCKED   		0x00002000 	// set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD   		0x00004000 	// set when the read thread is running [global]
#define DEV_V2FEATURES		0x00008000	// set for devices with V2 feature sets [global]
#define DEV_HIRES			0x00010000	// set for HiRes devices [global]

// Feature bits for the feature DWORD 
#define DEFAULT_FEATURES	0x00000000
#define HAS_BIDIR_RAMPS		0x00000001
#define HAS_PROFILES		0x00000002
#define HAS_HIRES			0x00000004
#define HAS_4CHANNELS		0x00000008
#define HAS_8CHANNELS		0x00000010
#define HAS_LONG_PROFILE	0x00000020
#define HAS_MCHANNELS		0x00000040

void fnLPS_Init(void);

void fnLPS_SetTraceLevel(int tracelevel, int IOtracelevel, bool verbose);
void fnLPS_SetTestMode(bool testmode);
int fnLPS_GetNumDevices();
int fnLPS_GetDevInfo(DEVID *ActiveDevices);
int fnLPS_GetModelName(DEVID deviceID, char *ModelName);
int fnLPS_InitDevice(DEVID deviceID);
int fnLPS_CloseDevice(DEVID deviceID);
int fnLPS_GetSerialNumber(DEVID deviceID);
int fnLPS_GetLibVersion();
int fnLPS_GetDeviceStatus(DEVID deviceID);

LVSTATUS fnLPS_SetChannel(DEVID deviceID, int channel);
LVSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency);
LVSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase);
LVSTATUS fnLPS_SetPhaseAngleQ(DEVID deviceID, int phase, int channel);

LVSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart);
LVSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop);
LVSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep);
LVSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2);
LVSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2);
LVSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime);
LVSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime);

LVSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phase);
LVSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount);
LVSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime);
LVSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLPS_StartProfile(DEVID deviceID, int mode);
LVSTATUS fnLPS_StartProfileMC(DEVID deviceID, int mode, int chmask, bool delayed);

LVSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up);
LVSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode);
LVSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable);
LVSTATUS fnLPS_StartRamp(DEVID deviceID, bool go);
LVSTATUS fnLPS_StartRampMC(DEVID deviceID, int mode, int chmask, bool deferred);

LVSTATUS fnLPS_SaveSettings(DEVID deviceID);


int fnLPS_GetWorkingFrequency(DEVID deviceID);
int fnLPS_GetMinWorkingFrequency(DEVID deviceID);
int fnLPS_GetMaxWorkingFrequency(DEVID deviceID);

int fnLPS_GetPhaseAngle(DEVID deviceID);
int fnLPS_GetRampStart(DEVID deviceID);
int fnLPS_GetRampEnd(DEVID deviceID);
int fnLPS_GetDwellTime(DEVID deviceID);
int fnLPS_GetDwellTimeTwo(DEVID deviceID);
int fnLPS_GetIdleTime(DEVID deviceID);
int fnLPS_GetHoldTime(DEVID deviceID);

int fnLPS_GetPhaseAngleStep(DEVID deviceID);
int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID);

int fnLPS_GetProfileElement(DEVID deviceID, int index);
int fnLPS_GetProfileCount(DEVID deviceID);
int fnLPS_GetProfileDwellTime(DEVID deviceID);
int fnLPS_GetProfileIdleTime(DEVID deviceID);
int fnLPS_GetProfileIndex(DEVID deviceID);

int fnLPS_GetMaxPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseShift(DEVID deviceID);
int fnLPS_GetMinPhaseStep(DEVID deviceID);
int fnLPS_GetFeatures(DEVID deviceID);
int fnLPS_GetNumChannels(DEVID deviceID);
int fnLPS_GetProfileMaxLength(DEVID deviceID);

char* fnLPS_perror(LVSTATUS status);
